clear all
set type double
set matsize 5000
set maxvar 32000
set more off
clear

* Assumes initial directory path is set to Codes folder
cd ".."
cap ssc install freduse
cap mkdir Temp
cap mkdir Output_Files

local K_it_types `""ipe" "ipp""' 
local K_nit_types `""ent" "ieq" "oeq" "rad" "str" "teq""'

*******************************
** Load Data
*******************************

foreach tab in "tauinc" "tauss" "tauk" "tauc" "taux" {
	import excel using "Input_Files/Taxes/McDaniel Tax Rates.xlsx", sheet("`tab'") firstrow clear
	keep A UnitedStates
	replace UnitedStates = UnitedStates[_n-1] if missing(UnitedStates)
	rename (A UnitedStates) (year `tab')
	save Temp/`tab', replace
}
foreach tab in "tauinc" "tauss" "tauk" "tauc" "taux" {
	merge 1:1 year using Temp/`tab', nogen keep(1 2 3)
}
gen taun = tauinc+tauss
drop tauinc tauss
save Temp\taus, replace

cap {
	freduse AAA DGS10 POP, clear
	gen year = year(daten)
	collapse (mean) valuer_US10=DGS10 valuer_AAA=AAA valuePOP=POP, by(year) 
	gen valuer_EQ = 5+valuer_US10
	reshape long value, i(year) j(kn_code) string
	drop if missing(value)
	save Temp\r, replace
}
	
insheet using Input_Files\NIPA\GDP\TablesRegister.txt, names clear
save Temp\TablesRegister, replace

insheet using Input_Files\NIPA\GDP\SeriesRegister.txt, names clear
save Temp\SeriesRegister, replace

insheet using Input_Files\NIPA\GDP\nipadataA.txt, names clear
destring period value, force replace ignore(",")
save Temp\nipadataA, replace

insheet using Input_Files\NIPA\FixedAssets\Section2All_csv.csv, clear
destring v1, force replace
drop if missing(v1)
gen table_counter = 1 if v1==1
gen table = sum(table_counter)
drop table_counter
tostring table, replace force
replace table = "2."+table
order table
rename (v1 v2 v3) (line desc code)
reshape long v, i(table line desc code) j(year)
replace year = year+1921
destring v, force replace ignore(",")
rename v value
save Temp\FA2, replace

* NOTE: We are combining two vintages of capital series (A, covering 1929-1996, and B, covering 1997 onward). There appear to be no jumps.
insheet using Input_Files\NIPA\FixedAssets\Section7All_csv.csv, clear
gen table = substr(v1,7,4) if substr(v1,1,5)=="Table"
replace table = table[_n-1] if missing(table)
destring v1, force replace
drop if missing(v1)
rename (v1 v2 v3) (line desc code)
destring v* line, force replace ignore(",")
reshape long v, i(table line desc code) j(year)
replace year = year+1921 if substr(table,-1,1)=="A"
replace year = year+1993 if substr(table,-1,1)=="B"
rename v value
order table desc line year
sort table line year
save Temp\FA7_unmerged, replace
drop if missing(value)
replace table = substr(table,1,3)
sort table line year
save Temp\FA7, replace

*******************************
** Get Investment Spend and Prices
*******************************

/* 
NIPA Table 5.3.4 has Investment Prices, NIPA Table 5.3.5 has Investment Spending:
	i) ICT includes the sum of:
		a) Private, Nonresidential, Equipment, Information processing equipment (Line 10)
		b) Private, Nonresidential, Intellectual property products, Software (Line 17)
	ii) Non-ICT (implicitly non-residential) includes the sum of:
		a) Private, Nonresidential, Structures (Line 3)
		b) Private, Nonresidential, Equipment, Industrial equipment (Line 13)
		c) Private, Nonresidential, Equipment, Transportation equipment (Line 14)
		d) Private, Nonresidential, Equipment, Other equipment (Line 15)
		e) Private, Nonresidential, Intellectual property products, Research and development (Line 18)
		f) Private, Nonresidential, Intellectual property products, Entertainment, literary, and artistic originals (Line 19)
	iii) Housing includes the sum of:
		a) Private, Residential (Line 21)
*/

foreach table in "T50304" "T50305" {
	if "`table'"=="T50304" {
		local varname = "xi"
	}
	if "`table'"=="T50305" {
		local varname = "X"
	}
	use Temp\SeriesRegister, clear
	gen keeper = .
	gen kn_code = ""
	foreach line_str in "10" "17" "3" "13" "14" "15" "18" "19" "21" {
		replace keeper = 1 if strmatch(tableidlineno,"*`table':`line_str'") | strmatch(tableidlineno,"*`table':`line_str'|*")
	}
	keep if keeper == 1
	drop keeper
	replace kn_code = "`varname'_it_ipe" if strmatch(tableidlineno,"*`table':10*")
	replace kn_code = "`varname'_it_ipp" if strmatch(tableidlineno,"*`table':17*")
	replace kn_code = "`varname'_nit_str" if strmatch(tableidlineno,"*`table':3*")
	replace kn_code = "`varname'_nit_ieq" if strmatch(tableidlineno,"*`table':13*")
	replace kn_code = "`varname'_nit_teq" if strmatch(tableidlineno,"*`table':14*")
	replace kn_code = "`varname'_nit_oeq" if strmatch(tableidlineno,"*`table':15*")
	replace kn_code = "`varname'_nit_rad" if strmatch(tableidlineno,"*`table':18*")
	replace kn_code = "`varname'_nit_ent" if strmatch(tableidlineno,"*`table':19*")
	replace kn_code = "`varname'_res" if strmatch(tableidlineno,"*`table':21*")
	keep seriescode kn_code
	merge 1:m seriescode using Temp\nipadataA, keep(3) keepusing(period value) nogen
	if "`table'"=="T50304" {
		replace value = value/100
	}
	rename period year
	keep kn_code year value
	sort kn_code year
	cd Temp
	save `varname'.dta, replace
	cd ..
}

/*
NIPA Table 5.9.4B has Investment Prices, NIPA Table 5.9.5B has Investment Prices
	iv) Government is line 1
*/		

foreach table in "T50904B" "T50905B" {
	if "`table'"=="T50904B" {
		local varname = "xi"
	}
	if "`table'"=="T50905B" {
		local varname = "X"
	}
	use Temp\SeriesRegister, clear
	gen keeper = .
	gen kn_code = ""
	foreach line_str in "1" {
		replace keeper = 1 if strmatch(tableidlineno,"*`table':`line_str'") | strmatch(tableidlineno,"*`table':`line_str'|*")
	}
	keep if keeper == 1
	drop keeper
	replace kn_code = "`varname'_gov" if strmatch(tableidlineno,"*`table':1*")
	keep seriescode kn_code
	merge 1:m seriescode using Temp\nipadataA, keep(3) keepusing(period value) nogen
	if "`table'"=="T50904B" {
		replace value = value/100
	}
	rename period year
	keep kn_code year value
	sort kn_code year
	cd Temp
	save `varname'_gov.dta, replace
	cd ..
}


*******************************
** Get Capital Stocks, Prices, Depreciation Rates
*******************************
/*
Fixed Asset Table 2.1 has Capital Stocks
	i) ICT includes the sum of:
		a) Private, Nonresidential, Equipment, Information processing equipment (Line 4)
		b) Private, Nonresidential, Intellectual property products, Software (Line 78)
	ii) Non-ICT (implicitly non-residential) includes the sum of:
		a) Private, Nonresidential, Structures (Line 36)
		b) Private, Nonresidential, Equipment, Industrial equipment (Line 11)
		c) Private, Nonresidential, Equipment, Transportation equipment (Line 18)
		d) Private, Nonresidential, Equipment, Other equipment (Line 26)
		e) Private, Nonresidential, Intellectual property products, Research and development (Line 82)
		f) Private, Nonresidential, Intellectual property products, Entertainment, literary, and artistic originals (Line 98)
	iii) Housing includes the sum of:
		a) Private, Residential (Line 67)
*/

foreach table_str in "2.1" "2.2" "2.4" {
	if "`table_str'"=="2.1" {
		local varname = "pKK"
	}
	if "`table_str'"=="2.2" {
		local varname = "K"
	}
	if "`table_str'"=="2.4" {
		local varname = "pKdeltaK"
	}
	use Temp\FA2 if table=="`table_str'", clear
	gen kn_code = ""
	replace kn_code = "`varname'_it_ipe" if line==4
	replace kn_code = "`varname'_it_ipp" if line==78
	replace kn_code = "`varname'_nit_str" if line==36
	replace kn_code = "`varname'_nit_ieq" if line==11
	replace kn_code = "`varname'_nit_teq" if line==18
	replace kn_code = "`varname'_nit_oeq" if line==26
	replace kn_code = "`varname'_nit_rad" if line==82
	replace kn_code = "`varname'_nit_ent" if line==98
	replace kn_code = "`varname'_res" if line==67
	drop if missing(kn_code) | missing(value)
	if "`table_str'"=="2.1" | "`table_str'"=="2.4" {
		replace value = value*1000
	}
	keep kn_code year value
	cd Temp
	save `varname'.dta, replace
	cd ..
}	

/*
NIPA Table 7.1-7.3 has for Non-Private Fixed Assets
	iv) Government is line 1
*/

	use Temp\FA7, clear
gen kn_code = ""
replace kn_code = "pKK_gov" if table=="7.1" & line==1
replace kn_code = "K_gov" if table=="7.2" & line==1
replace kn_code = "pKdeltaK_gov" if table=="7.3" & line==1
drop if missing(kn_code) | missing(value)
replace value = value*1000 if kn_code=="pKK_gov" | kn_code=="pKdeltaK_gov"
keep kn_code year value
cd Temp
foreach var in "pKK_gov" "K_gov" "pKdeltaK_gov" {
	preserve
	keep if kn_code=="`var'"
	save `var'.dta, replace
	restore
}
cd ..

use Temp/K, clear
append using Temp/pKK
append using Temp/K_gov
append using Temp/pKK_gov
reshape wide value, i(year) j(kn_code) string
foreach captype in "it" "nit" {
	foreach K_`captype'_type in `K_`captype'_types'  {
		gen ratio_tmp = valuepKK_`captype'_`K_`captype'_type'/valueK_`captype'_`K_`captype'_type' if year==2009
		sum ratio_tmp
		local ratio = `r(mean)'
		replace valueK_`captype'_`K_`captype'_type' = `ratio' * valueK_`captype'_`K_`captype'_type'
		drop ratio*
	}
}
foreach captype in "res" "gov" {
	gen ratio_tmp = valuepKK_`captype'/valueK_`captype' if year==2009
	sum ratio_tmp
	local ratio = `r(mean)'
	replace valueK_`captype' = `ratio' * valueK_`captype'
	drop ratio*
}
keep year valueK*
reshape long value, i(year) j(kn_code) string
save Temp/K_scaled, replace


*******************************
** Get GDP, GDP Deflator, Housing Services, Employment, Population, Hours
*******************************

use Temp\SeriesRegister, clear
gen keeper = .
gen kn_code = ""
replace kn_code = "pY" if strmatch(tableidlineno,"*T10504:1") | strmatch(tableidlineno,"*T10504:1|*")
replace kn_code = "pYY" if strmatch(tableidlineno,"*T10505:1") | strmatch(tableidlineno,"*T10505:1|*")
replace kn_code = "pC" if strmatch(tableidlineno,"*T10504:2") | strmatch(tableidlineno,"*T10504:2|*")
replace kn_code = "pCC" if strmatch(tableidlineno,"*T10505:2") | strmatch(tableidlineno,"*T10505:2|*")
replace kn_code = "pH" if strmatch(tableidlineno,"*T10504:16") | strmatch(tableidlineno,"*T10504:16|*")
replace kn_code = "pHH" if strmatch(tableidlineno,"*T10505:16") | strmatch(tableidlineno,"*T10505:16|*")
replace kn_code = "pG" if strmatch(tableidlineno,"*T10504:52") | strmatch(tableidlineno,"*T10504:52|*")
replace kn_code = "pGG" if strmatch(tableidlineno,"*T10505:52") | strmatch(tableidlineno,"*T10505:52|*")
replace kn_code = "WN" if strmatch(tableidlineno,"*T20100:2") | strmatch(tableidlineno,"*T20100:2|*")
replace kn_code = "WN_gov" if strmatch(tableidlineno,"*T20100:5") | strmatch(tableidlineno,"*T20100:5|*")
replace kn_code = "FTE" if strmatch(tableidlineno,"*T60500A:1") | strmatch(tableidlineno,"*T60500A:1|*")
replace kn_code = "FTE_gov" if strmatch(tableidlineno,"*T60500A:75") | strmatch(tableidlineno,"*T60500A:75|*")
replace kn_code = "Hours" if strmatch(tableidlineno,"*T60900B:1") | strmatch(tableidlineno,"*T60900B:1|*")
replace kn_code = "Hours_gov" if strmatch(tableidlineno,"*T60900B:20") | strmatch(tableidlineno,"*T60900B:20|*")
replace kn_code = "TaxProd" if strmatch(tableidlineno,"*T11000:7") | strmatch(tableidlineno,"*T11000:7|*")
replace kn_code = "Subs" if strmatch(tableidlineno,"*T11000:8") | strmatch(tableidlineno,"*T11000:8|*")
replace kn_code = "PropInc" if strmatch(tableidlineno,"*T11000:13") | strmatch(tableidlineno,"*T11000:13|*")
replace kn_code = "CorpVA" if strmatch(tableidlineno,"*T11400:1") | strmatch(tableidlineno,"*T11400:1|*")
replace kn_code = "CorpComp" if strmatch(tableidlineno,"*T11400:4") | strmatch(tableidlineno,"*T11400:4|*")
keep seriescode kn_code
drop if missing(kn_code)
merge 1:m seriescode using Temp\nipadataA, keep(3) keepusing(period value) nogen
rename period year
foreach price in "pY" "pC" "pH" "pG" {
	replace value = value/100 if kn_code=="`price'"
}
keep kn_code year value
sort kn_code year
save Temp\Misc.dta, replace


*******************************
** Create Final Data
*******************************

clear
cd Temp
foreach file in "xi" "X" "xi_gov" "X_gov" "pKK" "pKK_gov" "K_scaled" "pKdeltaK" "pKdeltaK_gov" "Misc" "r" {
	append using `file'
}
cd ..
reshape wide value, i(year) j(kn_code) string
renpfix value
sort year
save Temp\Combined_Raw.dta, replace

* Create IT and NonIT (plus Residential at end) for Capital Stocks, Investment, and Depreciation
foreach captype in "it" "nit" {
	gen pKK_`captype' = 0
	gen X_`captype' = 0
	gen pKdeltaK_`captype' = 0
	foreach K_`captype'_type in `K_`captype'_types'  {
		foreach var in "pKK" "X" "pKdeltaK" {
			replace `var'_`captype' = `var'_`captype'+`var'_`captype'_`K_`captype'_type'
		}
	}
	foreach K_`captype'_type in `K_`captype'_types'  {
		gen pK_`captype'_`K_`captype'_type' = pKK_`captype'_`K_`captype'_type'/K_`captype'_`K_`captype'_type'
			gen pK_`captype'_`K_`captype'_type'_growth = pK_`captype'_`K_`captype'_type'/pK_`captype'_`K_`captype'_type'[_n-1]
		gen pKK_`captype'_`K_`captype'_type'_share = pKK_`captype'_`K_`captype'_type'/pKK_`captype'
		gen xi_`captype'_`K_`captype'_type'_growth = xi_`captype'_`K_`captype'_type'/xi_`captype'_`K_`captype'_type'[_n-1]
		gen X_`captype'_`K_`captype'_type'_share = X_`captype'_`K_`captype'_type'/X_`captype'
		gen delta_`captype'_`K_`captype'_type' = pKdeltaK_`captype'_`K_`captype'_type'/pKK_`captype'_`K_`captype'_type'
	}
	gen pK_`captype'_growth = 0
	gen xi_`captype'_growth = 0
	foreach K_`captype'_type in `K_`captype'_types'  {
		replace pK_`captype'_growth = pK_`captype'_growth + 0.5*(pKK_`captype'_`K_`captype'_type'_share+pKK_`captype'_`K_`captype'_type'_share[_n-1])*pK_`captype'_`K_`captype'_type'_growth
		replace xi_`captype'_growth = xi_`captype'_growth + 0.5*(X_`captype'_`K_`captype'_type'_share+X_`captype'_`K_`captype'_type'_share[_n-1])*xi_`captype'_`K_`captype'_type'_growth
	}
	gen pK_`captype' = 1 if year==2009
	gen xi_`captype' = 1 if year==2009
	gen delta_`captype' = pKdeltaK_`captype'/pKK_`captype'
	gsort -year
	replace pK_`captype' = pK_`captype'[_n-1]/pK_`captype'_growth[_n-1] if missing(pK_`captype')
	replace xi_`captype' = xi_`captype'[_n-1]/xi_`captype'_growth[_n-1] if missing(xi_`captype')
	sort year
	replace pK_`captype' = pK_`captype'[_n-1]*pK_`captype'_growth if missing(pK_`captype')
	replace xi_`captype' = xi_`captype'[_n-1]*xi_`captype'_growth if missing(xi_`captype')
	gen K_`captype' = pKK_`captype'/pK_`captype'
}
foreach captype in "res" "gov" {
	gen pK_`captype' = pKK_`captype'/K_`captype'
	gen delta_`captype' = pKdeltaK_`captype'/pKK_`captype'
}

* Create Prices for GDP less Housing (and less Gov't) and Consumption less Housing

* YexH = GDP less Housing
gen pYexHYexH = pYY - pHH
gen pYexHYexH_share = pYexHYexH / pYY
gen pHH_share = pHH / pYY
gen pY_growth = pY/pY[_n-1]
gen pH_growth = pH/pH[_n-1]
gen pYexH_growth = (pY_growth - 0.5*(pHH_share+pHH_share[_n-1])*pH_growth) / (0.5*(pYexHYexH_share + pYexHYexH_share[_n-1]))
gen pYexH = 1 if year==2009
gsort -year 
replace pYexH = pYexH[_n-1]/pYexH_growth[_n-1] if missing(pYexH)
sort year
replace pYexH = pYexH[_n-1]*pYexH_growth if missing(pYexH)

* YexG = GDP less Government
gen pYexGYexG = pYY - pGG
gen pYexGYexG_share = pYexGYexG / pYY
gen pGG_share = pGG / pYY
replace pY_growth = pY/pY[_n-1]
gen pG_growth = pG/pG[_n-1]
gen pYexG_growth = (pY_growth - 0.5*(pGG_share+pGG_share[_n-1])*pG_growth) / (0.5*(pYexGYexG_share + pYexGYexG_share[_n-1]))
gen pYexG = 1 if year==2009
gsort -year 
replace pYexG = pYexG[_n-1]/pYexG_growth[_n-1] if missing(pYexG)
sort year
replace pYexG = pYexG[_n-1]*pYexG_growth if missing(pYexG)

* YexHexG = GDP less Housing less Government
gen pYexHexGYexHexG = pYexHYexH - pGG
gen pYexHexGYexHexG_share = pYexHexGYexHexG / pYexHYexH
replace pGG_share = pGG / pYexHYexH
replace pG_growth = pG/pG[_n-1]
gen pYexHexG_growth = (pYexH_growth - 0.5*(pGG_share+pGG_share[_n-1])*pG_growth) / (0.5*(pYexHexGYexHexG_share + pYexHexGYexHexG_share[_n-1]))
gen pYexHexG = 1 if year==2009
gsort -year 
replace pYexHexG = pYexHexG[_n-1]/pYexHexG_growth[_n-1] if missing(pYexHexG)
sort year
replace pYexHexG = pYexHexG[_n-1]*pYexHexG_growth if missing(pYexHexG)

* CexH = Consumption less Housing
gen pCexHCexH = pCC - pHH
gen pCexHCexH_Cshare = pCexHCexH / pCC
gen pHH_Cshare = pHH / pCC
gen pC_growth = pC/pC[_n-1]
gen pCexH_growth = (pC_growth - 0.5*(pHH_Cshare+pHH_Cshare[_n-1])*pH_growth) / (0.5*(pCexHCexH_Cshare + pCexHCexH_Cshare[_n-1]))
gen pCexH = 1 if year==2009
gsort -year 
replace pCexH = pCexH[_n-1]/pCexH_growth[_n-1] if missing(pCexH)
sort year
replace pCexH = pCexH[_n-1]*pCexH_growth if missing(pCexH)

drop if missing(pYY)
drop *_it_* *_nit_* *_growth pKdeltaK* *share pKK*
merge 1:1 year using Temp\taus, nogen keep(1 3)
sort year
gen sL_Corp = CorpComp/CorpVA
gen ambiginc = TaxProd-Subs+PropInc
drop CorpComp CorpVA TaxProd Subs PropInc
order year pY pYY pC pCC pH pHH pG pGG pYexH pYexHYexH pYexG pYexGYexG pYexHexG pYexHexGYexHexG pCexH pCexHCexH pK_it K_it pK_nit K_nit pK_res K_res pK_gov K_gov xi_it X_it xi_nit X_nit xi_res X_res xi_gov X_gov delta_it delta_nit delta_res WN WN_gov FTE FTE_gov Hours_gov r_AAA r_US10 r_EQ POP sL_Corp ambiginc 
!rmdir "Temp" /s /q 
save Output_Files/KN_MA_Input_Data.dta, replace
cd Codes






